# Copyright Amazon.com, Inc. or its affiliates.
# SPDX-License-Identifier: GPL-3.0 OR Apache-2.0

"""
This module is intended to enable and disable https repos on boot time
via cloud-config. This will be calling 'amazon-linux-https' script 
which is present on an instance via system-release.

**Internal name:** ``cc_amazonlinux_repo_https``

**Module frequency:** per instance

*Config keys**::

    amazonlinux_repo_https: [true|false] 
"""

import os

from cloudinit import log as logging
from cloudinit.settings import PER_INSTANCE
from cloudinit import util

LOG = logging.getLogger(__name__)

frequency = PER_INSTANCE

def handle(name, cfg, cloud, log, args):
    """
    amazonlinux_repo_https is for enabling or disabling https repos via
    cloud-config

    amazonlinux_repo_https: [true|false]
    """

    if not os.path.isfile("/usr/sbin/amazon-linux-https"):
        LOG.debug("Could not find /usr/sbin/amazon-linux-https "
                  "This image doesn't support HTTPS repositories. "
                  "Update system-release to 2.13 or newer ")
        return

    if "amazonlinux_repo_https" not in cfg:
        LOG.debug("Didn't find 'amazonlinux_repo_https' in configuration. "
                  "Defaulting to HTTP ")
    
    value  = util.get_cfg_option_bool(cfg, "amazonlinux_repo_https", False)

    if value:
        util.subp(['amazon-linux-https', 'enable'])
    else:
        util.subp(['amazon-linux-https', 'disable'])
